package info.kramann.fuzzy.fuzzyset;


/**
 * Beinhaltet zwei Eingangsfuzzysets und ein Ausgangsfuzzyset.
 * Dient zur Auswertung.
 * 
 * Die Regeln:
 * Jede mögliche Fuzzygröße A wird mit jeder möglichen B UND-verknüpft.
 * mapping_regeln legt fest, welcher Ausgangsfuzzygröße die jeweilige Regel zugeordnet ist.
 * Zeile: A, Spalte: B
 */
public class Fuzzysystem
{
     private double[] punkte_in_A;
     private double[] punkte_in_B;
     private double[] punkte_out;
     private int[][] mapping_regeln;
    
     private Eingangsfuzzyset eA;
     private Eingangsfuzzyset eB;
     private Ausgangsfuzzyset oX;
     
     public Fuzzysystem(double[] punkte_in_A, double[] punkte_in_B, double[] punkte_out, int[][] mapping_regeln)
     {
         this.punkte_in_A = punkte_in_A;
         this.punkte_in_B = punkte_in_B;
         this.punkte_out = punkte_out;
         this.mapping_regeln = mapping_regeln;
         
         eA = new Eingangsfuzzyset(punkte_in_A);
         eB = new Eingangsfuzzyset(punkte_in_B);
         oX = new Ausgangsfuzzyset(punkte_out);
     }
     
     public double auswerten(double xa, double xb)
     {
         double[] aktivA = eA.berechneZugehoerigkeitswerte(xa);
         double[] aktivB = eB.berechneZugehoerigkeitswerte(xb);
         
         System.out.println("Zugehörigkeitswerte von Eingang A:");
         for(int i=0;i<aktivA.length;i++)
             System.out.print(aktivA[i]+" ");
         System.out.println();    
         
         System.out.println("Zugehörigkeitswerte von Eingang B:");
         for(int i=0;i<aktivB.length;i++)
             System.out.print(aktivB[i]+" ");
         System.out.println();    
         
         double[] aktivierung = new double[punkte_out.length];
                  
         //Mapping durchgehen und UND-Verknüpfungen druchführen:
         for(int i=0;i<mapping_regeln.length;i++)
         {
             for(int k=0;k<mapping_regeln[i].length;k++)
             {    
                //Bei fehlenden Regeln negative Zahl in Mapping:
                if(mapping_regeln[i][k]>=0)
                {
                  if(aktivA[i]<aktivB[k])
                      aktivierung[mapping_regeln[i][k]]+=aktivA[i];
                  else    
                      aktivierung[mapping_regeln[i][k]]+=aktivB[k];
                }      
             }
         }
         
         System.out.println("Aktivierung der einzelnen Ausgangsgrößen:");
         for(int i=0;i<aktivierung.length;i++)
             System.out.print(aktivierung[i]+" ");
         System.out.println();    
                  
         return oX.berechneSchwerpunkt(aktivierung);
     }
     
     /**
      * Beispiel an der Tafel!
      */
     public static void main(String[] args)
     {
         double[] punkte_in_A = {1.0,2.0};
         double[] punkte_in_B = {1.0,2.0};
         double[] punkte_out = {1.0,2.0};
         int[][] mapping_regeln = {
                                      {-1,0},
                                      {-1,1}
                                  };         
      
         //REGELN:
         //WENN f1 klein UND f2 groß DANN z klein
         //WENN f1 groß  UND f2 groß DANN z groß
         
         //f1   f2  R1   R2   z
         //1    2   1    0    1
         //1.5  1.5 0.5  0.5  1.5
         //2    1   0    0    ?
         //1.25 2   0.75 0.25 1.25
         Fuzzysystem fs = new Fuzzysystem(punkte_in_A,punkte_in_B,punkte_out,mapping_regeln);                                  
         
         
         //Tests:
         double[][] tests = {
                                {1.0,2.0},
                                {1.5,1.5},
                                {2.0,1.0},
                                {1.25,2.0}
                            };
                            
         System.out.println("Testweise Auswertungen");                   
         for(int i=0;i<tests.length;i++)
         {
             System.out.println("************ "+i+".: *****************");
             double erg = fs.auswerten(tests[i][0],tests[i][1]);
             System.out.println("f1="+tests[i][0]+" f2="+tests[i][1]+" z="+erg);
         }
     }
}
