package fuzzy;

/**
 *  <h4>FuzzyRegler.java</h4>
 *  <br/>
 *  Copyright (C) 2011 Guido Kramann<br/>
 *  kramann@fh-brandenburg.de<br/>
 *  http://www.kramann.info<br/>
 *  <br/>
 *  <p>  
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *  </p>
 *  <p>
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *  </p>
 *  <p>
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *  </p>
 */

public class FuzzyRegler
{
    public FuzzyEingangsSet[] fes = null;
    private FuzzyAusgangsSet   fas = null;
    private int[][] regeln;   

    public void setFuzzyEingangsSet(FuzzyEingangsSet[] fes)
    {
        this.fes = fes;
    }

    public void setFuzzyAusgangsSet(FuzzyAusgangsSet fas)
    {
        this.fas = fas;
    }

    public void setFuzzyRegeln(int[][] regeln)
    {
        this.regeln = regeln;
    }

    public double berechneAusgang(double[] eingang)
    {
        double ergebnis = 0.0;
        double gesamtflaeche = 0.0;
        for(int i=0;i<regeln.length;i++)
        {
             double aktivierung = 1.0; //zunchst auf Maximalwert setzen.

             //Alle Eingangsgren der aktuellen Regel durchgehen
             for(int k=0;k<(regeln[i].length-1)/2;k++) 
             {
                 //Minimum der Zugehrigkeitsgrade finden:
                 double neue_aktivierung = 
                 fes[regeln[i][1+k*2]].getZugehoerigkeitsgrad(regeln[i][2+k*2],eingang[regeln[i][1+k*2]]);                
                 if(neue_aktivierung < aktivierung)
                     aktivierung = neue_aktivierung;
             }

             //Mit gefundenem Aktivierungsgrad der aktuellen Regel in das Ausgangs-Fuzzy-Set gehen und
             //dort daraus resultierende Flche und Schwerpunktlage bercksichtigen:
             double flaeche = fas.getFlaeche(regeln[i][0],aktivierung);             
             ergebnis+=flaeche*fas.getSchwerpunkt(regeln[i][0],aktivierung);
             gesamtflaeche+=flaeche;
        }
        return ergebnis/gesamtflaeche;
    }

    //Umsetzung des Lfter-Beispiels zur Kontrolle:
    public static void main(String[] args)
    {
        //1. Fuzzy-Regler erstellen:
        FuzzyRegler fuzzyregler = new FuzzyRegler(); 

        //2. Fuzzy-Eingangs-Sets erstellen:
        FuzzyEingangsSet[] fes = new FuzzyEingangsSet[2];

        FuzzyEingangsSet   raumtemperatur   = new FuzzyEingangsSet();
        raumtemperatur.erzeugeFuzzyGroessen(2);
        raumtemperatur.verteileFuzzyGroessenGleichmaessig(10.0,30.0);

        FuzzyEingangsSet   luftfeuchtigkeit = new FuzzyEingangsSet();
        luftfeuchtigkeit.erzeugeFuzzyGroessen(2);
        luftfeuchtigkeit.verteileFuzzyGroessenGleichmaessig(50.0,70.0);

        fes[0] = raumtemperatur;        
        fes[1] = luftfeuchtigkeit;        

        fuzzyregler.setFuzzyEingangsSet(fes);

        //3. Fuzzy-Ausgangs-Set erstellen:
        FuzzyAusgangsSet fas = new FuzzyAusgangsSet();
        fas.erzeugeFuzzyGroessen(3+2); //Drei Fuzzygren und Minimum und Maximum
        fas.verteileFuzzyGroessenGleichmaessig(-100.0,300.0);
       
        fuzzyregler.setFuzzyAusgangsSet(fas);

        //4. Fuzzy-Regeln hinzufgen:
        int[][] regeln = new int[4][];
        regeln[0] = new int[] {0, 0,0, 1,0};
        regeln[1] = new int[] {1, 0,0, 1,1};
        regeln[2] = new int[] {2, 0,1, 1,0};
        regeln[3] = new int[] {2, 0,1, 1,1};

        fuzzyregler.setFuzzyRegeln(regeln);

        //Einige Ausgangsgren zum Test bestimmen:
        System.out.println("Eingang: T=25 A=60, Sollausgang: 144.737, es ergibt sich:"
        +fuzzyregler.berechneAusgang(new double[] {25.0,60.0}));

        System.out.println("Eingang: T=50 A=100, Sollausgang: 200, es ergibt sich:"
        +fuzzyregler.berechneAusgang(new double[] {50.0,100.0}));

        System.out.println("Eingang: T=20 A=60, Sollausgang: 125, es ergibt sich:"
        +fuzzyregler.berechneAusgang(new double[] {20.0,60.0}));

        System.out.println("Eingang: T=25 A=70, Sollausgang: 168.182, es ergibt sich:"
        +fuzzyregler.berechneAusgang(new double[] {25.0,70.0}));
    }
}


